#include "Point.hpp"

Point::Point(int dimension) :
	coordinates(dimension),
	dimension(dimension)
{
}

Point::Point(Point const * point) :
	coordinates(point->coordinates),
	dimension(point->dimension)
{
}

Point::Point(std::vector<double> const & coordinates) :
	coordinates(coordinates),
	dimension(coordinates.size())
{
}

double & Point::operator[] (int index)
{
	return coordinates[index];
}

double const & Point::operator[] (int index) const
{
	return coordinates[index];
}

Point & Point::operator+=(const Point & point)
{
	if(this->getDimension() != point.getDimension())
		throw "Incompatible dimensions";
	for(size_t i = 0; i < point.getDimension(); ++i)
		this->coordinates[i] += point[i];
	return *this;
}

Point & Point::operator-=(const Point & point)
{
	if(this->getDimension() != point.getDimension())
		throw "Incompatible dimensions";
	for(size_t i = 0; i < point.getDimension(); ++i)
		this->coordinates[i] -= point[i];
	return *this;
}

bool Point::operator==(Point const & x) const
{
	return this->coordinates == x.coordinates;
}

bool Point::operator!=(Point const & x) const
{
	return !operator==(x);
}

Point const Point::operator+(Point const & x) const
{
	Point result(*this);
	result += x;
	return result;
}

Point const Point::operator-(Point const & x) const
{
	Point result(*this);
	result -= x;
	return result;
}

std::vector<double>::const_iterator Point::cbegin() const
{
	return coordinates.cbegin();
}

std::vector<double>::const_iterator Point::cend() const
{
	return coordinates.cend();
}

int Point::getDimension() const
{
	return dimension;
}

Point operator*(double s, Point const & x)
{
	Point y(x);
	for(size_t i = 0; i < x.getDimension(); ++i)
		y[i] *= s;
	return y;
}

double operator*(Point const & x, Point const & y)
{
	if(x.getDimension() != y.getDimension())
		throw "Incompatible dimensions";
	double scalprod = 0;
	for(size_t i = 0; i < x.getDimension(); ++i)
		scalprod += x[i] * y[i];
	return scalprod;
}

std::ostream& operator<<(std::ostream & stream, Point const & point)
{
	stream << "(";
	int i = 1;
	for(auto it = point.cbegin(); it != point.cend(); ++it)
	{
		stream << *it;
		if(i < point.getDimension())
			stream << ",";
		++i;
	}
	stream << ")";
	return stream;
}
