#ifndef LLOYDMEDIAN_HPP
#define	LLOYDMEDIAN_HPP

#include <algorithm>
#include <utility>
#include <memory>

#include "Metric.hpp"
#include "WeightedPoint.hpp"
#include "AdaptiveSampling.hpp"
#include "Weiszfeld.hpp"
#include "CenterOfGravity.hpp"
#include "KumarMedian.hpp"
#include "KumarMedian.hpp"

/**
 * @brief Adaption of Lloyd's algorithm for k-median
 * 
 * Uses k-means++-like sampling
 */
class LloydMedian
{
private:
    AdaptiveSampling sampling;
    Weiszfeld weiszfeld;
    CenterOfGravity centerOfGravity;
    KumarMedian kumar;
    Metric<Point>* metric;

    int weiszfeldMaxIt;
    int kumarMedianIterations;
public:
    LloydMedian(std::function<Metric<Point>*() > createMetric, std::function<Norm<Point>*() > createNorm);

    /**
     * Computes a center set
     * @param begin Input point set: begin
     * @param end Input point set: end
     * @param output Output iterator
     * @param k Number of centers
     * @param maxIterations Maximum number of iterations
     * @param n Size of input set (optional)
     */
    template<typename ForwardIterator, typename OutputIterator>
    void computeCenterSet(ForwardIterator begin, ForwardIterator end, OutputIterator output, size_t k, size_t maxIterations, size_t n = 0);

    virtual ~LloydMedian();
};

template<typename ForwardIterator, typename OutputIterator>
void LloydMedian::computeCenterSet(ForwardIterator begin, ForwardIterator end, OutputIterator output, size_t k, size_t maxIterations, size_t n)
{
    if (n == 0)
        for (ForwardIterator it = begin; it != end; ++it)
            ++n;
    int dimension = begin->getDimension();

    std::unique_ptr < std::vector < Point >> initialCenters = sampling.computeCenterSet(begin, end, k, n);

    std::vector<Point> centers(*initialCenters);
    std::vector<size_t> centerAssignmentIndices(n, 0);
    bool assignmentChanged = true;
    for (size_t i = 0; i < maxIterations && assignmentChanged; ++i)
    {
        assignmentChanged = false;
        std::vector < std::vector < WeightedPoint >> centerAssignments(k, std::vector<WeightedPoint>());
        size_t p = 0;
        for (ForwardIterator it = begin; it != end; ++it)
        {
            double minDist = std::numeric_limits<double>::infinity();
            size_t minCenter = 0;
            for (size_t c = 0; c < centers.size(); ++c)
            {
                double tmpDist = metric->distance(centers[c], *it);
                if (tmpDist < minDist)
                {
                    minDist = tmpDist;
                    minCenter = c;
                }
            }
            if (centerAssignmentIndices[p] != minCenter)
            {
                centerAssignmentIndices[p] = minCenter;
                assignmentChanged = true;
            }
            centerAssignments[minCenter].push_back(*it);
            ++p;
        }

        for (size_t c = 0; c < centers.size(); ++c)
        {
            if (centerAssignments[c].size() > 0)
            {
#ifdef KMEANS
                centers[c] = centerOfGravity.cog(centerAssignments[c].begin(), centerAssignments[c].end());
#else
                try
                {
                    centers[c] = weiszfeld.approximateOneMedian(centerAssignments[c].begin(), centerAssignments[c].end(), weiszfeldMaxIt);
                }
                catch (Weiszfeld::IterationFailed err)
                {
                    centers[c] = kumar.approximateOneMedianRounds(centerAssignments[c].begin(), centerAssignments[c].end(), 0.9999999999, weiszfeldMaxIt);
                }
#endif
            }
            else
                centers[c] = Point(dimension);
        }
    }

    for (size_t i = 0; i < centers.size(); ++i)
    {
        *output = centers[i];
        ++output;
    }
}

#endif	/* LLOYDMEDIAN_HPP */
