#include <memory>
#include <iostream>
#include <sstream>
#include <iomanip>
#include <vector>
#include <iterator>
#include <fstream>
#include <numeric>

#ifdef ERRORHANDLER
#include <cstdlib>
#include <stdexcept>
#include <execinfo.h>
#endif

#include <boost/algorithm/string.hpp>

#include "../probi-source/LloydMedian.hpp"
#include "../probi-source/LloydProbMedian.hpp"
#include "../probi-source/PKMedian.hpp"
#include "../probi-source/WeightedPoint.hpp"
#include "../probi-source/Point.hpp"
#include "../probi-source/ProbabilisticPoint.hpp"
#include "../probi-source/EuclideanMetric.hpp"
#include "../probi-source/EuclideanNorm.hpp"
#include "../probi-source/EuclideanSquaredMetric.hpp"
#include "../probi-source/EuclideanSquaredNorm.hpp"

#include "common.hpp"

using namespace std;

int main(int argc, char** argv)
{
#ifdef ERRORHANDLER
    std::set_terminate(handler);
#endif

    if (argc < 6)
    {
        std::cout << "Usage: k splitChar numOfPointsToMerge isWeighted maxLloydIterations maxLloydRounds seed file" << std::endl;
        std::cout << "e.g. : 10 S 10 1 100 3 879323 file.dat" << std::endl;
        return 1;
    }

    // k argument
    int k = atoi(argv[1]);
    // SplitChar argument
    char splitChar = *argv[2];
    if (splitChar == 'S')
        splitChar = ' ';
    std::stringstream splitStream;
    splitStream << splitChar;
    std::string splitSeq(splitStream.str());
    // Number of points to merge
    int numOfPointsToMerge = atoi(argv[3]);
    // Weight flag
    int int_isWeighted = atoi(argv[4]);
    bool isWeighted = (int_isWeighted == 1);
    // Maximum iterations
    int maxIt = atoi(argv[5]);
    // Maximum rounds of Lloyd
    int maxRounds = atoi(argv[6]);
    // Random seed
    std::mt19937* random = Randomness::getMT19937();
    new (random) std::mt19937(atoi(argv[7]));
    // Input file
    std::fstream filestr(argv[8], std::fstream::in);

#ifdef KMEANS
    LloydProbMedian lloydprob([]()
    {
        return new EuclideanSquaredMetric();
    }, []()
    {
        return new EuclideanSquaredNorm();
    });
    PKMedian pkmed([]()
    {
        return new EuclideanSquaredMetric();
    });
#else
    LloydProbMedian lloydprob([]()
    {
        return new EuclideanMetric();
    }, []()
    {
        return new EuclideanNorm();
    });
    PKMedian pkmed([]()
    {
        return new EuclideanMetric();
    });
#endif
    
    std::vector<ProbabilisticPoint> ppoints;
    double prob = 1 / double(numOfPointsToMerge);

    // Data dimension
    int dim = 0;
    std::function<bool(ProbabilisticPoint) > determineDimension = [&dim] (ProbabilisticPoint pp)
      {
          dim = pp[0].getDimension();
          return false;
      };
    filestr.clear();
    filestr.seekg(0);
    processStream(filestr, determineDimension, splitSeq, numOfPointsToMerge, prob, isWeighted);

    // Read input file to RAM
    std::function<bool(ProbabilisticPoint) > processProbPoint = [&ppoints] (ProbabilisticPoint pp)
      {
        ppoints.push_back(pp);
        return true;
      };
    filestr.clear();
    filestr.seekg(0);
    processStream(filestr, processProbPoint, splitSeq, numOfPointsToMerge, prob, isWeighted);

    // Compute centers
    std::vector<Point> centers;
    double costs = std::numeric_limits<double>::infinity();
    for(int i = 0; i < maxRounds; ++i)
    {
      std::vector<Point> tmpCenters;
      lloydprob.computeCenterSet(ppoints.begin(), ppoints.end(), std::back_inserter(tmpCenters), k, maxIt, ppoints.size());
      double tmpCosts = pkmed.weightedCost(ppoints.begin(), ppoints.end(), tmpCenters.begin(), tmpCenters.end());
      if(tmpCosts < costs)
      {
        costs = tmpCosts;
        centers = tmpCenters;
      }
    }
    
    // Output centers
    for(int i = 0; i < centers.size(); ++i)
    {
      Point& p = centers[i];
      for(int j = 0; j < p.getDimension(); ++j)
      {
        std::cout << p[j];
        if(j < p.getDimension()-1)
          std::cout << ",";
      }
        std::cout << "\n";
    }

    return 0;
}

